import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:font_awesome_flutter/font_awesome_flutter.dart';
import 'package:wordpress_app/blocs/config_bloc.dart';
import 'package:wordpress_app/utils/toast.dart';
import '../blocs/user_bloc.dart';
import '../models/user.dart';
import '../services/auth_service.dart';

class SocialLogins extends ConsumerStatefulWidget {
  const SocialLogins({super.key, required this.afterSignIn});
  final VoidCallback afterSignIn;

  @override
  ConsumerState<SocialLogins> createState() => _SocialLoginsState();
}

class _SocialLoginsState extends ConsumerState<SocialLogins> {
  bool googleLoginStarted = false;
  bool fbLoginStarted = false;
  bool appleLoginStarted = false;

  Future<void> _handleGoogleSignIn() async {
    final ub = ref.read(userBlocProvider);
    setState(() => googleLoginStarted = true);
    UserModel? user = await AuthService.signInWithGoogle();
    if (user != null) {
      debugPrint(user.toJson().toString());
      ub
          .guestUserSignout()
          .then((_) => ub.saveUserData(user, 'google'))
          .then((_) => ub.setSignIn())
          .then((_) {
        setState(() => googleLoginStarted = true);
        widget.afterSignIn();
      });
    } else {
      await AuthService.googleSignIn.signOut();
      setState(() => googleLoginStarted = false);
      openToast('Error on Google Login. Please try again!');
      debugPrint('google login error');
    }
  }

  Future<void> _handleFacebookSignIn() async {
    final ub = ref.read(userBlocProvider);
    setState(() => fbLoginStarted = true);
    UserModel? user = await AuthService.signInWithFacebook();
    if (user != null) {
      debugPrint(user.toJson().toString());
      ub
          .guestUserSignout()
          .then((_) => ub.saveUserData(user, 'fb'))
          .then((_) => ub.setSignIn())
          .then((_) {
        setState(() => fbLoginStarted = true);
        widget.afterSignIn();
      });
    } else {
      await AuthService.facebookAuth.logOut();
      setState(() => fbLoginStarted = false);
      openToast('Error on Facebook Login. Please try again!');
      debugPrint('fb login error');
    }
  }

  Future<void> _handleAppleSignIn() async {
    final ub = ref.read(userBlocProvider);
    setState(() => appleLoginStarted = true);
    UserModel? user =
        await AuthService.signInWithApple().onError((error, stackTrace) {
      setState(() => appleLoginStarted = false);
      return null;
    });
    if (user != null) {
      debugPrint(user.toJson().toString());
      ub
          .guestUserSignout()
          .then((_) => ub.saveUserData(user, 'apple'))
          .then((_) => ub.setSignIn())
          .then((_) {
        setState(() => appleLoginStarted = true);
        widget.afterSignIn();
      });
    } else {
      setState(() => appleLoginStarted = false);
      openToast('Error on Apple Login. Please try again!');
      debugPrint('apple login error');
    }
  }

  @override
  Widget build(BuildContext context) {
    return Padding(
      padding: const EdgeInsets.only(top: 20, bottom: 0),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.start,
        children: [
          InkWell(
            onTap: () => _handleGoogleSignIn(),
            child: CircleAvatar(
              radius: 25,
              backgroundColor: Theme.of(context).colorScheme.primaryContainer,
              child: googleLoginStarted
                  ? CircularProgressIndicator(
                      color: Theme.of(context).colorScheme.onPrimaryContainer,
                    )
                  : Icon(
                      FontAwesomeIcons.google,
                      size: 25,
                      color: Theme.of(context).colorScheme.onPrimaryContainer,
                    ),
            ),
          ),
          Visibility(
            visible: ref.read(configBlocProvider).configs!.fbLoginEnabled,
            child: Row(
              children: [
                const SizedBox(
                  width: 10,
                ),
                InkWell(
                  onTap: () => _handleFacebookSignIn(),
                  child: CircleAvatar(
                    radius: 25,
                    backgroundColor:
                        Theme.of(context).colorScheme.secondaryContainer,
                    child: fbLoginStarted
                        ? CircularProgressIndicator(
                            color: Theme.of(context)
                                .colorScheme
                                .onSecondaryContainer,
                          )
                        : Icon(
                            FontAwesomeIcons.facebookF,
                            size: 25,
                            color: Theme.of(context)
                                .colorScheme
                                .onSecondaryContainer,
                          ),
                  ),
                ),
              ],
            ),
          ),
          const SizedBox(
            width: 10,
          ),
          Visibility(
            // visible: Platform.isIOS,
            child: InkWell(
              onTap: () => _handleAppleSignIn(),
              child: CircleAvatar(
                radius: 25,
                child: appleLoginStarted
                    ? CircularProgressIndicator(
                        color: Theme.of(context).colorScheme.onSurfaceVariant,
                      )
                    : Icon(
                        FontAwesomeIcons.apple,
                        size: 28,
                        color: Theme.of(context).colorScheme.onSurfaceVariant,
                      ),
              ),
            ),
          ),
        ],
      ),
    );
  }
}
